#!/usr/bin/env python3
"""
===============================================================================
 BITCOIN PROVENANCE TOOL (With Master Ledger)
===============================================================================
Usage:
  python3 Bitcoin_Provenance_Tool.py              (Menu & Status Browser)
  python3 Bitcoin_Provenance_Tool.py <filename>   (Direct Action)

Updates:
  - Generates individual artifacts (file.provenance.json)
  - Maintains a MASTER LEDGER (master_ledger.json) organizing all records.
  - Full tab-completion and state tracking.
===============================================================================
"""

import os
import sys
import json
import hashlib
import subprocess
import shutil
import re
import glob
import readline
from datetime import datetime, timezone

# ---------------------------------------------------------------------------
# 0. CONFIGURATION
# ---------------------------------------------------------------------------
PROOF_DIR = "bitcoin_proofs"
IDENTITY_FILENAME = "identity.key"
LEDGER_FILENAME = "master_ledger.json"

# Colors
C_RESET  = "\033[0m"
C_CYAN   = "\033[1;36m"
C_GREEN  = "\033[1;32m"
C_YELLOW = "\033[1;33m"
C_RED    = "\033[1;31m"
C_BOLD   = "\033[1m"
C_GREY   = "\033[90m"

# ---------------------------------------------------------------------------
# 1. SETUP: TAB AUTOCOMPLETE
# ---------------------------------------------------------------------------
def path_completer(text, state):
    if '~' in text: text = os.path.expanduser(text)
    return [x for x in glob.glob(text + '*')][state]

readline.set_completer_delims(' \t\n;')
readline.parse_and_bind("tab: complete")
readline.set_completer(path_completer)

# ---------------------------------------------------------------------------
# 2. DEPENDENCY CHECKS
# ---------------------------------------------------------------------------
def check_requirements():
    missing = []
    try:
        import nacl.signing
    except ImportError:
        missing.append("pynacl")

    ots_path = shutil.which("ots")
    if not ots_path:
        user_bin = os.path.expanduser("~/.local/bin/ots")
        if os.path.exists(user_bin):
            ots_path = user_bin
        else:
            missing.append("opentimestamps-client")

    if missing:
        print(f"{C_RED}❌ MISSING REQUIREMENTS{C_RESET}")
        print(f"   pip install {' '.join(missing)} --user --break-system-packages")
        sys.exit(1)
    return ots_path

OTS_EXEC = check_requirements()
from nacl.signing import SigningKey
from nacl.encoding import HexEncoder

# ---------------------------------------------------------------------------
# 3. HELPER FUNCTIONS
# ---------------------------------------------------------------------------
def get_provenance_filename(target_file):
    base = os.path.basename(target_file)
    return os.path.join(PROOF_DIR, f"{base}.provenance.json")

def update_master_ledger(record_entry):
    """Updates the central master_ledger.json file."""
    ledger_path = os.path.join(PROOF_DIR, LEDGER_FILENAME)
    ledger_data = {"records": {}}

    # Load existing
    if os.path.exists(ledger_path):
        try:
            with open(ledger_path, 'r') as f:
                ledger_data = json.load(f)
        except json.JSONDecodeError:
            pass

    # Update record (keyed by filename for uniqueness)
    fname = record_entry['target_file']['filename']
    ledger_data['records'][fname] = {
        "status": record_entry['provenance_status'],
        "last_updated": record_entry['last_updated'],
        "block_height": record_entry['blockchain_data']['block_height'],
        "note": record_entry.get('user_note', ''),
        "sha256": record_entry['target_file']['sha256']
    }

    # Save
    with open(ledger_path, 'w') as f:
        json.dump(ledger_data, f, indent=2)

    print(f"   📘 Ledger updated: {ledger_path}")

def show_file_listing():
    print(f"\n{C_CYAN}📁 AVAILABLE FILES:{C_RESET}")
    print("-" * 70)

    # Check Ledger first for quick status
    ledger_path = os.path.join(PROOF_DIR, LEDGER_FILENAME)
    ledger_records = {}
    if os.path.exists(ledger_path):
        try:
            with open(ledger_path, 'r') as f:
                ledger_records = json.load(f).get("records", {})
        except: pass

    try:
        files = [f for f in os.listdir('.') if os.path.isfile(f) and not f.startswith('.')]
        files.sort()
    except OSError:
        files = []

    if not files:
        print("   (No files found)")

    for f in files:
        # Check if in ledger
        if f in ledger_records:
            rec = ledger_records[f]
            status = rec.get('status')
            if status == "confirmed":
                icon = "✅"
                display = f"{C_GREEN}{f} {C_GREY}[Block {rec.get('block_height')}]{C_RESET}"
            else:
                icon = "⏳"
                display = f"{C_YELLOW}{f} {C_GREY}[Pending]{C_RESET}"
        else:
            # Fallback to checking file existence if not in ledger
            json_path = get_provenance_filename(f)
            if os.path.exists(json_path):
                 icon = "⏳"
                 display = f"{C_YELLOW}{f} {C_GREY}[Anchored/Unindexed]{C_RESET}"
            else:
                icon = "📄"
                display = f"{f}"

        print(f" {icon} {display}")

    print("-" * 70)

def load_key():
    if not os.path.exists(PROOF_DIR): os.makedirs(PROOF_DIR)

    key_path = os.path.join(PROOF_DIR, IDENTITY_FILENAME)

    if os.path.exists(key_path):
        with open(key_path, "r") as f:
            return SigningKey(f.read().strip(), encoder=HexEncoder), IDENTITY_FILENAME
    else:
        print(f"\n{C_YELLOW}✨ Generating NEW Identity Key...{C_RESET}")
        sk = SigningKey.generate()
        with open(key_path, "w") as f:
            f.write(sk.encode(encoder=HexEncoder).decode())
        print(f"   Saved to: {key_path}")
        return sk, IDENTITY_FILENAME

def get_hash(filepath):
    print(f"{C_CYAN}⚙️  Hashing {os.path.basename(filepath)}...{C_RESET}")
    sha256, sha512 = hashlib.sha256(), hashlib.sha512()
    total = os.path.getsize(filepath)
    processed = 0
    with open(filepath, "rb") as f:
        while chunk := f.read(16*1024*1024):
            sha256.update(chunk)
            sha512.update(chunk)
            processed += len(chunk)
            if total > 0: print(f"   {int((processed/total)*100)}%", end="\r")
    print("   ✅ Hashing Complete.\n")
    return sha256.hexdigest(), sha512.hexdigest()

# ---------------------------------------------------------------------------
# 4. WORKFLOW: NEW ANCHOR
# ---------------------------------------------------------------------------
def run_new_anchor(target_file, json_path):
    print("\n" + "="*60)
    print(f"{C_BOLD} 🆕 NEW FILE DETECTED{C_RESET}")
    print("="*60)

    note = input(f" Enter a Note for the Blockchain (e.g. 'v1.0') [Enter to skip]: ").strip()

    # 1. Crypto Operations
    sk, key_filename = load_key()
    pk_hex = sk.verify_key.encode(encoder=HexEncoder).decode()
    h256, h512 = get_hash(target_file)

    # Sign Hash + Note
    signature = sk.sign(f"{h512}|{note}".encode()).signature.hex()

    # 2. Define Artifact Filenames
    base_name = os.path.basename(target_file)
    ots_filename = f"{base_name}.ots"
    ots_path = os.path.join(PROOF_DIR, ots_filename)

    # 3. Build State JSON
    manifest = {
        "version": "1.1",
        "provenance_status": "pending",
        "last_updated": datetime.now(timezone.utc).isoformat(),
        "target_file": {
            "filename": base_name,
            "size_bytes": os.path.getsize(target_file),
            "sha256": h256,
            "sha512": h512
        },
        "artifacts": {
            "proof_file": ots_filename,
            "key_file": key_filename,
            "json_file": os.path.basename(json_path)
        },
        "identity": {
            "public_key": pk_hex,
            "signature": signature,
            "signed_payload": "sha512|note"
        },
        "blockchain_data": {
            "block_height": None,
            "confirmed_at": None
        },
        "user_note": note
    }

    # 4. Write Artifact JSON
    with open(json_path, "w") as f:
        json.dump(manifest, f, indent=2)

    # 5. Update Master Ledger
    update_master_ledger(manifest)

    # 6. Stamp
    print(f"⏳ Submitting fingerprint to Bitcoin aggregators...")
    try:
        # Stamp the JSON manifest (Metadata + Hash linkage)
        subprocess.check_call([OTS_EXEC, "stamp", json_path])

        # OTS creates json_path.ots. Rename to clean name.
        generated_ots = json_path + ".ots"
        if os.path.exists(generated_ots):
            os.rename(generated_ots, ots_path)

        print(f"\n{C_GREEN}✅ ANCHOR SUBMITTED!{C_RESET}")
        print(f"   Artifacts in '{PROOF_DIR}/':")
        print(f"   1. {os.path.basename(json_path)} (Individual Record)")
        print(f"   2. {ots_filename} (Cryptographic Proof)")
        print(f"   3. {LEDGER_FILENAME} (Master Organizer)")
        print("-" * 60)
        print(f" {C_YELLOW}⏳ NEXT STEP:{C_RESET} Wait 12-24 hours for Bitcoin mining.")
        print("=" * 60)

    except Exception as e:
        print(f"{C_RED}❌ Error stamping: {e}{C_RESET}")

# ---------------------------------------------------------------------------
# 5. WORKFLOW: UPDATE & VERIFY (WITH GENESIS BACKUP)
# ---------------------------------------------------------------------------
def run_existing_verification(json_path):
    print("\n" + "="*60)
    print(f"{C_BOLD} 🔍 EXISTING PROVENANCE FOUND{C_RESET}")
    print("="*60)

    # 1. Load State
    with open(json_path, 'r') as f:
        manifest = json.load(f)

    proof_filename = manifest['artifacts']['proof_file']
    ots_path = os.path.join(PROOF_DIR, proof_filename)

    if not os.path.exists(ots_path):
        print(f"{C_RED}❌ Error: Proof file '{proof_filename}' missing.{C_RESET}")
        return

    # --- SAFETY: Work on a temporary copy ---
    temp_ots_path = ots_path + ".tmp"
    shutil.copy2(ots_path, temp_ots_path)

    try:
        # 2. Upgrade (Run on the TEMP file)
        print(" 📡 Checking for Bitcoin updates...")
        subprocess.run([OTS_EXEC, "upgrade", temp_ots_path], capture_output=True, text=True)

        # 3. Verify (Run on the TEMP file)
        verify_proc = subprocess.run([OTS_EXEC, "verify", temp_ots_path], capture_output=True, text=True)

        # Capture ALL output (stdout + stderr)
        output = verify_proc.stdout + verify_proc.stderr

        # 4. Analyze Output
        if "Bitcoin block" in output:
            match_block = re.search(r"Bitcoin block\s+(\d+)", output)
            match_time = re.search(r"attests existence as of (.+)", output)

            block = match_block.group(1) if match_block else "Unknown"

            # Clean up timestamp
            time_s = "Unknown"
            if match_time:
                time_s = match_time.group(1).split('\n')[0].strip()

            # Update JSON State
            manifest["provenance_status"] = "confirmed"
            manifest["last_updated"] = datetime.now(timezone.utc).isoformat()
            manifest["blockchain_data"]["block_height"] = block
            manifest["blockchain_data"]["confirmed_at"] = time_s

            with open(json_path, "w") as f:
                json.dump(manifest, f, indent=2)

            # Update Master Ledger
            update_master_ledger(manifest)

            # --- BACKUP STRATEGY: PRESERVE GENESIS ---
            genesis_backup = ots_path + ".genesis.bak"
            if not os.path.exists(genesis_backup):
                shutil.copy2(ots_path, genesis_backup)
                print(f"    📦 Genesis record saved to: {C_YELLOW}{os.path.basename(genesis_backup)}{C_RESET}")

            # Overwrite main file with upgraded version
            shutil.move(temp_ots_path, ots_path)

            # Cleanup temp artifacts
            if os.path.exists(temp_ots_path + ".bak"):
                os.remove(temp_ots_path + ".bak")

            print(f"\n {C_GREEN}✅ CONFIRMED!{C_RESET}")
            print(f"    File locked in Bitcoin Block: {C_CYAN}{block}{C_RESET}")
            print(f"    Timestamp: {C_CYAN}{time_s}{C_RESET}")

        elif "Pending" in output or "incomplete" in output:
            print(f"\n {C_YELLOW}⏳ STATUS: PENDING MINING{C_RESET}")
            print("    Bitcoin hasn't mined the transaction yet.")
            # Discard temp to keep state clean
            if os.path.exists(temp_ots_path):
                os.remove(temp_ots_path)

        else:
            print(f"\n {C_RED}⚠️  STATUS UNKNOWN{C_RESET}")
            print(f"{C_GREY}--- RAW OUTPUT ---{C_RESET}")
            print(output)
            print(f"{C_GREY}------------------{C_RESET}")
            if os.path.exists(temp_ots_path):
                os.remove(temp_ots_path)

    except Exception as e:
        print(f"\n{C_RED}❌ EXECUTION ERROR: {e}{C_RESET}")
        if os.path.exists(temp_ots_path):
            os.remove(temp_ots_path)

# ---------------------------------------------------------------------------
# 6. MAIN ENTRY
# ---------------------------------------------------------------------------
def main():
    if len(sys.argv) < 2:
        print(f"{C_BOLD}BITCOIN PROVENANCE TOOL v1.1{C_RESET}")
        show_file_listing()
        try:
            user_input = input(f"\n{C_YELLOW}Enter filename > {C_RESET}").strip().strip("'")
        except KeyboardInterrupt:
            print("\nExiting."); sys.exit(0)
        if not user_input: sys.exit(0)
        target_file = user_input
    else:
        target_file = sys.argv[1]

    if not os.path.exists(target_file):
        print(f"{C_RED}❌ Error: File '{target_file}' not found.{C_RESET}")
        sys.exit(1)

    json_path = get_provenance_filename(target_file)

    if os.path.exists(json_path):
        run_existing_verification(json_path)
    else:
        run_new_anchor(target_file, json_path)

if __name__ == "__main__":
    main()
